; Code written in IDL (Exelis: http://www.exelisvis.com/IntelliEarthSolutions/GeospatialProducts/IDL.aspx)
; Creates random receptor patterns in a double-parasegment, evaluates how robust each pattern is to 1, 2 or 3 missing cells, and outputs a robustness score for each generated pattern
; The double-parasegment is set up along the AP-axis with exactly 4 cells per parasegment.
; Constraints: 
; - each receptor type is expressed in exactly 4 cells per double parasegment
; - every AP edge of a receptor’s expression scores a ‘receptor difference’ point, which contributes to the robustness score.
; Further details in Tetley, Blanchard, Fletcher, Adams & Sanson, eLife, 2016 
; Last edited: 11th March 2016, Guy Blanchard, gb288@cam.ac.uk

pro Robust_GBE

  nTolls = 4
  Tolls = BYTARR(nTolls,8)
  NbrDiff = FLTARR(8)
  Nbr2Diff = FLTARR(8)
  Nbr3Diff = FLTARR(8)
  Nbr4Diff = FLTARR(8)

  nKeep = 20
  SavedTots = FLTARR(nKeep)
  SavedTolls = BYTARR(nTolls,8,nKeep)
  SavedNbrDiff = FLTARR(8,nKeep)
  SavedNbr2Diff = FLTARR(8,nKeep)
  SavedNbr3Diff = FLTARR(8,nKeep)
  SavedNbr4Diff = FLTARR(8,nKeep)

  Seed = !NULL
  nIts = 1000000L ;1M
  AllTots = FLTARR(nIts)
  
  for m=0L,nIts-1L do begin
    Tolls[*,*] = 0
    
    for t=0,nTolls-1 do begin
      for i=0,3 do begin ;allocate 4 per toll
        repeat begin
          f = RANDOMU(seed)
        endrep until ~Tolls[t,f*8]
        Tolls[t,f*8] = 1
      endfor
    endfor

    for i=0,7 do begin
      NbrDiff[i] = TOTAL(Tolls[*,i] NE Tolls[*,(i+1) MOD 8])
      Nbr2Diff[i] = TOTAL(Tolls[*,i] NE Tolls[*,(i+2) MOD 8])
      Nbr3Diff[i] = TOTAL(Tolls[*,i] NE Tolls[*,(i+3) MOD 8])
      Nbr4Diff[i] = TOTAL(Tolls[*,i] NE Tolls[*,(i+4) MOD 8])
    endfor

    Tot = TOTAL(NbrDiff NE 0) ;immediate neighbours
    Tot += TOTAL((Nbr2Diff GT NbrDiff) AND (Nbr2Diff GT SHIFT(NbrDiff,-1))) ;missing cells
    Tot += TOTAL((Nbr3Diff GT Nbr2Diff) AND (Nbr3Diff GT SHIFT(Nbr2Diff,-1))) ;2 missing cells
    Tot += TOTAL((Nbr4Diff GT Nbr3Diff) AND (Nbr4Diff GT SHIFT(Nbr3Diff,-1))) ;3 missing cells
    if Tot GT 24 then begin
      print, Tot,":"
      print, Tolls
    endif
    
    AllTots[m] = Tot
    
    Diffs = Tot - SavedTots
    mx = MAX(Diffs)
    if mx GE 0 then begin
      replace = -1
      idx = WHERE(Tot EQ SavedTots)
      if idx[0] EQ -1 then begin
        mn = MIN(SavedTots,replace)
      endif else begin
        for j=0,N_ELEMENTS(idx)-1 do begin
          for s=0,7 do begin
            if TOTAL(SHIFT(NbrDiff,s) EQ SavedNbrDiff[*,idx[j]]) EQ 8 AND TOTAL(SHIFT(Nbr2Diff,s) EQ SavedNbr2Diff[*,idx[j]]) EQ 8 then break
            if TOTAL(SHIFT(REVERSE(NbrDiff),s) EQ SavedNbrDiff[*,idx[j]]) EQ 8 AND TOTAL(SHIFT(REVERSE(Nbr2Diff),s) EQ SavedNbr2Diff[*,idx[j]]) EQ 8 then break
          endfor
          if s EQ 8 then begin
            replace = idx[j]
            break
          endif
        endfor
      endelse
      if replace NE -1 then begin
        SavedTots[replace] = Tot
        SavedTolls[*,*,replace] = Tolls
        SavedNbrDiff[*,replace] = NbrDiff
        SavedNbr2Diff[*,replace] = Nbr2Diff
        SavedNbr3Diff[*,replace] = Nbr3Diff
        SavedNbr4Diff[*,replace] = Nbr4Diff
        sidx = SORT(SavedTots)
        SavedTots = SavedTots[sidx]
        SavedTolls = SavedTolls[*,*,sidx]
        SavedNbrDiff = SavedNbrDiff[*,sidx]
        SavedNbr2Diff = SavedNbr2Diff[*,sidx]
        SavedNbr3Diff = SavedNbr3Diff[*,sidx]
        SavedNbr4Diff = SavedNbr4Diff[*,sidx]
      endif
    endif
  endfor
  for i=0,nKeep-1 do begin
    print, SavedTots[i]
    print, SavedTolls[*,*,i]
  endfor
  h1 = Histogram(AllTots,MIN=0, MAX=32,NBINS=33)
  p = BARPLOT(h1,YRANGE=[0,MAX(h1)*1.1],/YLOG,YTITLE='Freq',XTITLE='Robustness total')
  p.Save,"~/Desktop/RobustGBE.pdf"
  SAVE, FILENAME='~/Desktop/RobustGBE.sav',SavedTots,SavedTolls
end